% Heterogeneous Propagation Medium Example
%
% This example provides a demonstration of using k-wave for the simulation
% and detection of photoacoustic waves within a two-dimensional
% heterogeneous propagation medium. It builds on the Homogeneous
% Propagation Medium Example.
%
% author: Bradley Treeby
% date: 30th June 2009
% last update: 16th July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nx = 128;           % number of pixels in the x (column) direction
Nz = 128;           % number of pixels in the z (row) direction
dx = 0.1e-3;        % pixel width [m]
dz = 0.1e-3;        % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
c0 = 1500;                  % [m/s]
c = c0*ones(Nz, Nx);        % [m/s]
c(1:Nz/2, :) = c0*1.2;

rho0 = 1000;                % [kg/m^3]
rho = rho0*ones(Nz, Nx);    % [kg/m^3]
rho(:, Nx/4:end) = rho0*1.2;

% create initial pressure distribution using makeDisc
disc_magnitude = 5;
disc_x_pos = 50;    % pixels
disc_z_pos = 50;    % pixels
disc_radius = 8;    % pixels
disc_1 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

disc_magnitude = 3;
disc_x_pos = 60;    % pixels
disc_z_pos = 80;    % pixels
disc_radius = 5;    % pixels
disc_2 = disc_magnitude*makeDisc(Nx, Nz, disc_x_pos, disc_z_pos, disc_radius);

p0 = disc_1 + disc_2;

% define a centered circular sensor
sensor_radius = 4e-3;   % [m]
num_sensor_points = 50;
sensor_mask = makeCartCircle(sensor_radius, num_sensor_points);

% run the simulation setting the display flag to true
sensor_data = kspaceFirstOrder2D(p0, kgrid, c, rho, 'auto', sensor_mask, 'PlotLayout', true);

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the simulated sensor data
figure;
imagesc(sensor_data, [-1, 1]);
colormap(getColorMap);
ylabel('Sensor Position');
xlabel('Time Step');
colorbar;